/** @file   cgamebase.h
 * @brief   Declaration of CGameBase - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_CGAMEBASE_H
#define H_ENG2D_CGAMEBASE_H

#include <string>

namespace eng2d {

/** @class  CGameBase
 * @brief   A baseclass for fixed speed game engine.
 * @author  Tomi Lamminsaari
 *
 * This class runs the application at fixed speed.
 */
class CGameBase
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  

  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	CGameBase();


	/** Destructor
   */
	virtual ~CGameBase();

private:

	/** Copy constructor.
   * @param     rO                Reference to another CGameBase
   */
  CGameBase( const CGameBase& rO );

	/** Assignment operator
   * @param     rO                Reference to another CGameBase
   * @return    Reference to us.
   */
  CGameBase& operator = ( const CGameBase& rO );

public:


  ///
  /// Methods
  /// =======

  /** Initializes the game. This method should allocate the resources the
   * game needs.
   *
   * If any errors are encountered, you should provide descriptive errormessage
   * by calling the @c setErrorMessage(...) - method.
   * @return    Nonzero if errors occured.
   */
  virtual int initialize() = 0;
  
  /** This method should take care of updating the game structures and
   * objects. This method will be called as many times per second as
   * you told to the constructor as you called it.
   *
   * This method must not have any drawing code.
   */
  virtual void update() = 0;
  
  /** This method should take care of drawing the game contents.
   */
  virtual void redraw() = 0;

  /** This method must cleanup and release all the resources you have
   * allocated during the execution.
   */
  virtual void cleanup() = 0;
  
  
  /** Initialzes this baseclass structures and calls the @c initialize() -
   * method to initialize the child class as well.
   * @return    Nonzero if fails.
   */
  int init();
  
  /** Runs this application. This method has the code that controls the speed
   * of the game. This will call the @c update() and @c redraw() - methods
   * when needed.
   *
   * This method uses the @c quitNow() - method to find out when to quit.
   * The value received from that method will be returned.
   * @return    The value received from @c quitNow() - method. You can
   *            use this value to define different ending reasons.
   */
  int run();
  
  /** Releases the resources allocated for us. This method calls the @c cleanup()
   * method from our child class.
   */
  void flush();
  
  
  
  ///
  /// Getter methods
  /// ==============
  
  /** This method should tell when we should quit. The @c run() - method
   * calls this method to find out when we should quit the game.
   *
   * Every nonzero value makes the @c run() - method to exit.
   * @return    Nonzero if we're willing to quit.
   */
  virtual int quitNow() const = 0;
  
  /** Returns the possible error message.
   * @return    The errormessage.
   */
  std::string getErrorMessage() const;

protected:

  /** Sets the errormessage. Every consecutive calls concatenates the
   * given messages to each other. This method adds a newline character
   * to the end of string before appending the current error string.
   * @param     aMessage          The error message.
   */
  void setErrorMessage( const std::string& aMessage );



  ///
  /// Members
  /// =======

  /** Maximum number of frames we are allowed to skip. */
  int   iMaxFrameSkips;
  
  /** The errormessage */
  std::string iErrorMessage;


private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
